<?php
/**
 * SaaS AI Application Generator Installation Wizard
 * 
 * Author: MiniMax Agent
 * Date: 2025-11-07
 */

// Check if already installed
if (file_exists('../config/installed.lock')) {
    die('<div style="text-align:center;padding:50px;font-family:Arial;"><h2>Application Already Installed</h2><p>The application has already been installed. Please remove the install/ directory for security reasons.</p></div>');
}

session_start();

// Initialize installation steps
$steps = [
    1 => 'Welcome',
    2 => 'Requirements',
    3 => 'Database',
    4 => 'Configuration',
    5 => 'Complete'
];

$currentStep = isset($_GET['step']) ? (int)$_GET['step'] : 1;
$currentStep = max(1, min(5, $currentStep));

// Handle form submissions
$errors = [];
$success = false;

if ($_POST) {
    switch ($currentStep) {
        case 3:
            // Database configuration
            $dbHost = $_POST['db_host'] ?? 'localhost';
            $dbName = $_POST['db_name'] ?? '';
            $dbUser = $_POST['db_user'] ?? '';
            $dbPass = $_POST['db_pass'] ?? '';
            $dbPrefix = $_POST['db_prefix'] ?? 'saas_';
            
            if (empty($dbName) || empty($dbUser)) {
                $errors[] = 'Database name and username are required';
            } else {
                // Test database connection
                try {
                    $dsn = "mysql:host=$dbHost;dbname=$dbName;charset=utf8mb4";
                    $pdo = new PDO($dsn, $dbUser, $dbPass, [
                        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION
                    ]);
                    
                    // Store database config in session
                    $_SESSION['db_config'] = [
                        'host' => $dbHost,
                        'name' => $dbName,
                        'user' => $dbUser,
                        'pass' => $dbPass,
                        'prefix' => $dbPrefix
                    ];
                    
                    header('Location: ?step=4');
                    exit();
                } catch (PDOException $e) {
                    $errors[] = 'Database connection failed: ' . $e->getMessage();
                }
            }
            break;
            
        case 4:
            // Configuration settings
            $siteName = $_POST['site_name'] ?? 'AI App Generator SaaS';
            $siteUrl = $_POST['site_url'] ?? '';
            $adminEmail = $_POST['admin_email'] ?? '';
            $adminPassword = $_POST['admin_password'] ?? '';
            $adminPasswordConfirm = $_POST['admin_password_confirm'] ?? '';
            $companyName = $_POST['company_name'] ?? 'AI Solutions Ltd.';
            
            if (empty($adminEmail) || empty($adminPassword)) {
                $errors[] = 'Admin email and password are required';
            } elseif ($adminPassword !== $adminPasswordConfirm) {
                $errors[] = 'Passwords do not match';
            } elseif (strlen($adminPassword) < 8) {
                $errors[] = 'Password must be at least 8 characters';
            } elseif (!filter_var($adminEmail, FILTER_VALIDATE_EMAIL)) {
                $errors[] = 'Invalid email address';
            } else {
                // Store configuration in session
                $_SESSION['site_config'] = [
                    'site_name' => $siteName,
                    'site_url' => $siteUrl,
                    'admin_email' => $adminEmail,
                    'admin_password' => $adminPassword,
                    'company_name' => $companyName
                ];
                
                header('Location: ?step=5');
                exit();
            }
            break;
            
        case 5:
            // Complete installation
            try {
                // Create config file
                $configContent = "<?php\n";
                $configContent .= "// Generated configuration file\n";
                $configContent .= "define('DB_HOST', '" . $_SESSION['db_config']['host'] . "');\n";
                $configContent .= "define('DB_NAME', '" . $_SESSION['db_config']['name'] . "');\n";
                $configContent .= "define('DB_USER', '" . $_SESSION['db_config']['user'] . "');\n";
                $configContent .= "define('DB_PASS', '" . $_SESSION['db_config']['pass'] . "');\n";
                $configContent .= "define('SITE_NAME', '" . addslashes($_SESSION['site_config']['site_name']) . "');\n";
                $configContent .= "define('SITE_URL', '" . addslashes($_SESSION['site_config']['site_url']) . "');\n";
                $configContent .= "define('SITE_ADMIN_EMAIL', '" . addslashes($_SESSION['site_config']['admin_email']) . "');\n";
                $configContent .= "define('COMPANY_NAME', '" . addslashes($_SESSION['site_config']['company_name']) . "');\n";
                $configContent .= "?>";
                
                file_put_contents('../config/config.php', $configContent);
                
                // Create database
                $dsn = "mysql:host={$_SESSION['db_config']['host']};dbname={$_SESSION['db_config']['name']};charset=utf8mb4";
                $pdo = new PDO($dsn, $_SESSION['db_config']['user'], $_SESSION['db_config']['pass'], [
                    PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION
                ]);
                
                // Read and execute schema
                $schema = file_get_contents('../database/schema.sql');
                $statements = explode(';', $schema);
                
                foreach ($statements as $statement) {
                    $statement = trim($statement);
                    if (!empty($statement)) {
                        $pdo->exec($statement);
                    }
                }
                
                // Create admin user
                $adminPassword = password_hash($_SESSION['site_config']['admin_password'], PASSWORD_DEFAULT);
                $stmt = $pdo->prepare("INSERT INTO users (email, password, name, role, status, plan_id) VALUES (?, ?, ?, 'admin', 'active', 'enterprise')");
                $stmt->execute([
                    $_SESSION['site_config']['admin_email'],
                    $adminPassword,
                    'System Administrator'
                ]);
                
                // Mark as installed
                file_put_contents('../config/installed.lock', date('Y-m-d H:i:s'));
                
                $success = true;
                
                // Clear session
                session_destroy();
                
            } catch (Exception $e) {
                $errors[] = 'Installation failed: ' . $e->getMessage();
            }
            break;
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Installation - AI App Generator SaaS</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 20px;
        }
        
        .installation-container {
            background: white;
            border-radius: 20px;
            box-shadow: 0 20px 60px rgba(0,0,0,0.1);
            max-width: 600px;
            width: 100%;
            overflow: hidden;
        }
        
        .header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 30px;
            text-align: center;
        }
        
        .header h1 {
            font-size: 28px;
            margin-bottom: 10px;
        }
        
        .header p {
            opacity: 0.9;
            font-size: 16px;
        }
        
        .steps {
            display: flex;
            background: #f8f9fa;
            padding: 0;
        }
        
        .step {
            flex: 1;
            padding: 15px;
            text-align: center;
            font-size: 14px;
            color: #666;
            border-right: 1px solid #dee2e6;
            position: relative;
        }
        
        .step:last-child {
            border-right: none;
        }
        
        .step.active {
            background: #667eea;
            color: white;
        }
        
        .step.completed {
            background: #28a745;
            color: white;
        }
        
        .content {
            padding: 40px;
        }
        
        .form-group {
            margin-bottom: 25px;
        }
        
        label {
            display: block;
            margin-bottom: 8px;
            font-weight: 600;
            color: #333;
        }
        
        input[type="text"],
        input[type="email"],
        input[type="password"],
        input[type="url"] {
            width: 100%;
            padding: 12px 15px;
            border: 2px solid #e9ecef;
            border-radius: 8px;
            font-size: 16px;
            transition: border-color 0.3s;
        }
        
        input:focus {
            outline: none;
            border-color: #667eea;
        }
        
        .btn {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 12px 30px;
            border: none;
            border-radius: 8px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: transform 0.2s;
        }
        
        .btn:hover {
            transform: translateY(-2px);
        }
        
        .btn:disabled {
            opacity: 0.6;
            cursor: not-allowed;
            transform: none;
        }
        
        .btn-secondary {
            background: #6c757d;
            margin-left: 10px;
        }
        
        .errors {
            background: #f8d7da;
            color: #721c24;
            padding: 15px;
            border-radius: 8px;
            margin-bottom: 20px;
        }
        
        .errors ul {
            margin-left: 20px;
        }
        
        .success {
            background: #d4edda;
            color: #155724;
            padding: 20px;
            border-radius: 8px;
            text-align: center;
            margin-bottom: 20px;
        }
        
        .requirements {
            background: #f8f9fa;
            border-radius: 8px;
            padding: 20px;
            margin-bottom: 20px;
        }
        
        .requirement {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 10px 0;
            border-bottom: 1px solid #e9ecef;
        }
        
        .requirement:last-child {
            border-bottom: none;
        }
        
        .status {
            padding: 5px 15px;
            border-radius: 20px;
            font-size: 12px;
            font-weight: 600;
        }
        
        .status.pass {
            background: #d4edda;
            color: #155724;
        }
        
        .status.fail {
            background: #f8d7da;
            color: #721c24;
        }
        
        .footer {
            background: #f8f9fa;
            padding: 20px 40px;
            text-align: center;
            color: #6c757d;
            font-size: 14px;
        }
    </style>
</head>
<body>
    <div class="installation-container">
        <div class="header">
            <h1>AI App Generator SaaS</h1>
            <p>Installation Wizard</p>
        </div>
        
        <div class="steps">
            <?php foreach ($steps as $num => $name): ?>
                <div class="step <?= $num == $currentStep ? 'active' : ($num < $currentStep ? 'completed' : '') ?>">
                    <?= $num ?>. <?= $name ?>
                </div>
            <?php endforeach; ?>
        </div>
        
        <div class="content">
            <?php if (!empty($errors)): ?>
                <div class="errors">
                    <strong>Please fix the following errors:</strong>
                    <ul>
                        <?php foreach ($errors as $error): ?>
                            <li><?= htmlspecialchars($error) ?></li>
                        <?php endforeach; ?>
                    </ul>
                </div>
            <?php endif; ?>
            
            <?php if ($success): ?>
                <div class="success">
                    <h2>Installation Complete! 🎉</h2>
                    <p>Your AI App Generator SaaS platform has been successfully installed.</p>
                    <p><strong>Admin Credentials:</strong></p>
                    <p>Email: <?= htmlspecialchars($_SESSION['site_config']['admin_email'] ?? '') ?></p>
                    <p>Password: <?= htmlspecialchars($_SESSION['site_config']['admin_password'] ?? '') ?></p>
                    <br>
                    <a href="../?page=login" class="btn">Go to Login</a>
                </div>
            <?php else: ?>
                
                <?php if ($currentStep == 1): ?>
                    <!-- Welcome Step -->
                    <h2>Welcome to AI App Generator SaaS</h2>
                    <p>Thank you for choosing our AI-powered application generator platform. This wizard will help you set up your SaaS business in just a few steps.</p>
                    
                    <div class="requirements">
                        <h3>What you'll get:</h3>
                        <ul>
                            <li>✅ Complete SaaS platform</li>
                            <li>✅ User management system</li>
                            <li>✅ Subscription plans & billing</li>
                            <li>✅ AI application generation</li>
                            <li>✅ Admin dashboard</li>
                            <li>✅ Payment integration</li>
                            <li>✅ Marketing tools</li>
                        </ul>
                    </div>
                    
                    <a href="?step=2" class="btn">Start Installation</a>
                    
                <?php elseif ($currentStep == 2): ?>
                    <!-- Requirements Step -->
                    <h2>System Requirements</h2>
                    <p>Please ensure your server meets the following requirements:</p>
                    
                    <div class="requirements">
                        <?php
                        $requirements = [
                            'PHP Version >= 7.4' => version_compare(PHP_VERSION, '7.4.0', '>='),
                            'MySQL Extension' => extension_loaded('mysqli') || extension_loaded('pdo_mysql'),
                            'cURL Extension' => extension_loaded('curl'),
                            'JSON Extension' => extension_loaded('json'),
                            'OpenSSL Extension' => extension_loaded('openssl'),
                            'Write Permissions' => is_writable('../config/') && is_writable('../uploads/')
                        ];
                        
                        $allPassed = true;
                        foreach ($requirements as $req => $passed) {
                            if (!$passed) $allPassed = false;
                        }
                        ?>
                        
                        <?php foreach ($requirements as $req => $passed): ?>
                            <div class="requirement">
                                <span><?= htmlspecialchars($req) ?></span>
                                <span class="status <?= $passed ? 'pass' : 'fail' ?>">
                                    <?= $passed ? 'PASS' : 'FAIL' ?>
                                </span>
                            </div>
                        <?php endforeach; ?>
                    </div>
                    
                    <?php if ($allPassed): ?>
                        <p style="color: #28a745;"><strong>Great! Your server meets all requirements.</strong></p>
                        <a href="?step=3" class="btn">Continue to Database Setup</a>
                    <?php else: ?>
                        <p style="color: #721c24;"><strong>Please fix the requirements above before continuing.</strong></p>
                        <button class="btn" onclick="location.reload()">Check Again</button>
                    <?php endif; ?>
                    
                <?php elseif ($currentStep == 3): ?>
                    <!-- Database Step -->
                    <h2>Database Configuration</h2>
                    <p>Configure your MySQL database connection:</p>
                    
                    <form method="post">
                        <div class="form-group">
                            <label for="db_host">Database Host</label>
                            <input type="text" id="db_host" name="db_host" value="localhost" required>
                        </div>
                        
                        <div class="form-group">
                            <label for="db_name">Database Name</label>
                            <input type="text" id="db_name" name="db_name" placeholder="your_username_saas_ai_generator" required>
                        </div>
                        
                        <div class="form-group">
                            <label for="db_user">Database Username</label>
                            <input type="text" id="db_user" name="db_user" placeholder="your_username_saas_user" required>
                        </div>
                        
                        <div class="form-group">
                            <label for="db_pass">Database Password</label>
                            <input type="password" id="db_pass" name="db_pass" required>
                        </div>
                        
                        <div class="form-group">
                            <label for="db_prefix">Table Prefix (optional)</label>
                            <input type="text" id="db_prefix" name="db_prefix" value="saas_">
                            <small>Leave empty for no prefix</small>
                        </div>
                        
                        <button type="submit" class="btn">Test Connection</button>
                    </form>
                    
                <?php elseif ($currentStep == 4): ?>
                    <!-- Configuration Step -->
                    <h2>Site Configuration</h2>
                    <p>Configure your site settings and create admin account:</p>
                    
                    <form method="post">
                        <div class="form-group">
                            <label for="site_name">Site Name</label>
                            <input type="text" id="site_name" name="site_name" value="AI App Generator SaaS" required>
                        </div>
                        
                        <div class="form-group">
                            <label for="site_url">Site URL</label>
                            <input type="url" id="site_url" name="site_url" placeholder="https://yourdomain.com/saas-ai-generator" required>
                        </div>
                        
                        <div class="form-group">
                            <label for="company_name">Company Name</label>
                            <input type="text" id="company_name" name="company_name" value="AI Solutions Ltd." required>
                        </div>
                        
                        <div class="form-group">
                            <label for="admin_email">Admin Email</label>
                            <input type="email" id="admin_email" name="admin_email" required>
                        </div>
                        
                        <div class="form-group">
                            <label for="admin_password">Admin Password</label>
                            <input type="password" id="admin_password" name="admin_password" required>
                            <small>Minimum 8 characters</small>
                        </div>
                        
                        <div class="form-group">
                            <label for="admin_password_confirm">Confirm Password</label>
                            <input type="password" id="admin_password_confirm" name="admin_password_confirm" required>
                        </div>
                        
                        <button type="submit" class="btn">Complete Installation</button>
                    </form>
                    
                <?php elseif ($currentStep == 5): ?>
                    <!-- Complete Step -->
                    <h2>Ready to Complete Installation</h2>
                    <p>Click the button below to finish setting up your AI App Generator SaaS platform:</p>
                    
                    <form method="post">
                        <button type="submit" class="btn">Complete Installation</button>
                    </form>
                    
                <?php endif; ?>
            <?php endif; ?>
        </div>
        
        <div class="footer">
            <p>AI App Generator SaaS v1.0.0 | Installation Wizard</p>
        </div>
    </div>
</body>
</html>