import React, { createContext, useContext, useEffect, useState, useCallback } from 'react';
import { useNavigate } from 'react-router-dom';
import { apiClient } from '@/lib/api';
import { toast } from 'sonner';

interface User {
  id: number;
  username: string;
  email: string;
  full_name: string;
  role: 'admin' | 'teacher' | 'student';
  madrasa_id: number;
  phone?: string;
  avatar_url?: string;
  [key: string]: any; // For additional user data
}

interface AuthContextType {
  user: User | null;
  loading: boolean;
  isAuthenticated: boolean;
  login: (username: string, password: string) => Promise<void>;
  register: (userData: {
    username: string;
    email: string;
    password: string;
    full_name: string;
    role: 'admin' | 'teacher' | 'student';
    madrasa_id?: number;
  }) => Promise<void>;
  logout: () => void;
  updateProfile: (userData: { full_name?: string; phone?: string; email?: string }) => Promise<void>;
  changePassword: (currentPassword: string, newPassword: string) => Promise<void>;
  refreshUser: () => Promise<void>;
}

const AuthContext = createContext<AuthContextType | undefined>(undefined);

export const useAuth = () => {
  const context = useContext(AuthContext);
  if (context === undefined) {
    throw new Error('useAuth must be used within an AuthProvider');
  }
  return context;
};

interface AuthProviderProps {
  children: React.ReactNode;
}

export const AuthProvider: React.FC<AuthProviderProps> = ({ children }) => {
  const [user, setUser] = useState<User | null>(null);
  const [loading, setLoading] = useState(true);
  const navigate = useNavigate();

  const isAuthenticated = !!user;

  // Check if user is logged in on app start
  useEffect(() => {
    checkAuth();
  }, []);

  const checkAuth = async () => {
    try {
      if (apiClient.isAuthenticated()) {
        // Try to get current user profile
        const response = await apiClient.getProfile();
        if (response.success && response.data) {
          setUser(response.data);
        } else {
          // Token might be invalid
          apiClient.logout();
        }
      }
    } catch (error) {
      console.error('Auth check failed:', error);
      // Clear invalid token
      localStorage.removeItem('auth_token');
    } finally {
      setLoading(false);
    }
  };

  const login = useCallback(async (username: string, password: string) => {
    try {
      const response = await apiClient.login(username, password);
      
      if (response.success) {
        setUser(response.data.user);
        toast.success('Login successful');
        
        // Redirect based on role
        if (response.data.user.role === 'admin') {
          navigate('/dashboard');
        } else if (response.data.user.role === 'teacher') {
          navigate('/dashboard');
        } else if (response.data.user.role === 'student') {
          navigate('/dashboard');
        }
      } else {
        throw new Error(response.message || 'Login failed');
      }
    } catch (error) {
      console.error('Login error:', error);
      const message = error instanceof Error ? error.message : 'Login failed';
      toast.error(message);
      throw error;
    }
  }, [navigate]);

  const register = useCallback(async (userData: {
    username: string;
    email: string;
    password: string;
    full_name: string;
    role: 'admin' | 'teacher' | 'student';
    madrasa_id?: number;
  }) => {
    try {
      const response = await apiClient.register(userData);
      
      if (response.success) {
        toast.success('Registration successful! Please login with your credentials.');
        navigate('/auth');
      } else {
        throw new Error(response.message || 'Registration failed');
      }
    } catch (error) {
      console.error('Registration error:', error);
      const message = error instanceof Error ? error.message : 'Registration failed';
      toast.error(message);
      throw error;
    }
  }, [navigate]);

  const logout = useCallback(() => {
    try {
      // Logout from server
      apiClient.logout();
    } catch (error) {
      console.error('Logout error:', error);
    } finally {
      // Clear local state regardless of server response
      setUser(null);
      navigate('/auth');
      toast.success('Logged out successfully');
    }
  }, [navigate]);

  const updateProfile = useCallback(async (userData: {
    full_name?: string;
    phone?: string;
    email?: string;
  }) => {
    try {
      const response = await apiClient.updateProfile(userData);
      
      if (response.success) {
        // Refresh user data
        await refreshUser();
        toast.success('Profile updated successfully');
      } else {
        throw new Error(response.message || 'Profile update failed');
      }
    } catch (error) {
      console.error('Profile update error:', error);
      const message = error instanceof Error ? error.message : 'Profile update failed';
      toast.error(message);
      throw error;
    }
  }, []);

  const changePassword = useCallback(async (currentPassword: string, newPassword: string) => {
    try {
      const response = await apiClient.changePassword(currentPassword, newPassword);
      
      if (response.success) {
        toast.success('Password changed successfully');
      } else {
        throw new Error(response.message || 'Password change failed');
      }
    } catch (error) {
      console.error('Password change error:', error);
      const message = error instanceof Error ? error.message : 'Password change failed';
      toast.error(message);
      throw error;
    }
  }, []);

  const refreshUser = useCallback(async () => {
    try {
      const response = await apiClient.getProfile();
      if (response.success && response.data) {
        setUser(response.data);
      }
    } catch (error) {
      console.error('Profile refresh error:', error);
      // If profile refresh fails, logout
      if (apiClient.isAuthenticated()) {
        // Token might be invalid
        logout();
      }
    }
  }, [logout]);

  const value: AuthContextType = {
    user,
    loading,
    isAuthenticated,
    login,
    register,
    logout,
    updateProfile,
    changePassword,
    refreshUser,
  };

  return (
    <AuthContext.Provider value={value}>
      {children}
    </AuthContext.Provider>
  );
};

// Role-based access control utilities
export const useRole = () => {
  const { user } = useAuth();

  const isAdmin = user?.role === 'admin';
  const isTeacher = user?.role === 'teacher';
  const isStudent = user?.role === 'student';

  const canAccess = (requiredRole: 'admin' | 'teacher' | 'student') => {
    if (!user) return false;
    
    const roleHierarchy = { student: 1, teacher: 2, admin: 3 };
    const userLevel = roleHierarchy[user.role] || 0;
    const requiredLevel = roleHierarchy[requiredRole] || 0;
    
    return userLevel >= requiredLevel;
  };

  const canEdit = (ownerId: number) => {
    if (!user) return false;
    return user.id === ownerId || user.role === 'admin';
  };

  return {
    user,
    isAdmin,
    isTeacher,
    isStudent,
    canAccess,
    canEdit,
    role: user?.role || 'student',
  };
};

// Protected route component
interface ProtectedRouteProps {
  children: React.ReactNode;
  requiredRole?: 'admin' | 'teacher' | 'student';
}

export const ProtectedRoute: React.FC<ProtectedRouteProps> = ({ 
  children, 
  requiredRole = 'student' 
}) => {
  const { user, loading } = useAuth();
  const { canAccess } = useRole();

  if (loading) {
    return (
      <div className="flex items-center justify-center min-h-screen">
        <div className="animate-spin rounded-full h-32 w-32 border-b-2 border-green-600"></div>
      </div>
    );
  }

  if (!user) {
    return (
      <div className="flex items-center justify-center min-h-screen">
        <div className="text-center">
          <h1 className="text-2xl font-bold text-gray-900 mb-4">Access Denied</h1>
          <p className="text-gray-600 mb-4">Please login to access this page.</p>
          <button
            onClick={() => navigate('/auth')}
            className="bg-green-600 text-white px-4 py-2 rounded hover:bg-green-700"
          >
            Go to Login
          </button>
        </div>
      </div>
    );
  }

  if (requiredRole && !canAccess(requiredRole)) {
    return (
      <div className="flex items-center justify-center min-h-screen">
        <div className="text-center">
          <h1 className="text-2xl font-bold text-gray-900 mb-4">Insufficient Permissions</h1>
          <p className="text-gray-600 mb-4">
            You don't have permission to access this page.
          </p>
          <button
            onClick={() => navigate('/dashboard')}
            className="bg-green-600 text-white px-4 py-2 rounded hover:bg-green-700"
          >
            Go to Dashboard
          </button>
        </div>
      </div>
    );
  }

  return <>{children}</>;
};

export default AuthContext;