#!/bin/bash

# Madrasa Management System - Deployment Script
# © 2025 MiniMax Agent

echo "🏗️ Madrasa Management System - Deployment Started"
echo "=================================================="

# Colors for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

# Function to print colored output
print_status() {
    echo -e "${GREEN}[INFO]${NC} $1"
}

print_warning() {
    echo -e "${YELLOW}[WARNING]${NC} $1"
}

print_error() {
    echo -e "${RED}[ERROR]${NC} $1"
}

print_step() {
    echo -e "${BLUE}[STEP]${NC} $1"
}

# Check if required tools are installed
check_requirements() {
    print_step "Checking requirements..."
    
    if ! command -v php &> /dev/null; then
        print_error "PHP is not installed"
        exit 1
    fi
    
    if ! command -v mysql &> /dev/null; then
        print_warning "MySQL client not found - make sure MySQL is available"
    fi
    
    if ! command -v npm &> /dev/null; then
        print_error "Node.js/npm is not installed"
        exit 1
    fi
    
    print_status "All requirements satisfied"
}

# Database setup
setup_database() {
    print_step "Setting up database..."
    
    read -p "Enter MySQL root password: " -s DB_ROOT_PASSWORD
    echo
    
    read -p "Enter database name [madrasa_management]: " DB_NAME
    DB_NAME=${DB_NAME:-madrasa_management}
    
    read -p "Enter database username [root]: " DB_USER
    DB_USER=${DB_USER:-root}
    
    read -p "Enter database password for $DB_USER: " -s DB_PASSWORD
    echo
    
    # Create database and user
    mysql -u"$DB_USER" -p"$DB_ROOT_PASSWORD" << EOF
CREATE DATABASE IF NOT EXISTS $DB_NAME CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
CREATE USER IF NOT EXISTS '$DB_USER'@'localhost' IDENTIFIED BY '$DB_PASSWORD';
GRANT ALL PRIVILEGES ON $DB_NAME.* TO '$DB_USER'@'localhost';
FLUSH PRIVILEGES;
EOF
    
    if [ $? -eq 0 ]; then
        print_status "Database '$DB_NAME' created successfully"
    else
        print_error "Database creation failed"
        exit 1
    fi
    
    # Import schema
    print_step "Importing database schema..."
    mysql -u"$DB_USER" -p"$DB_PASSWORD" "$DB_NAME" < database/schema.sql
    
    if [ $? -eq 0 ]; then
        print_status "Database schema imported successfully"
    else
        print_error "Database schema import failed"
        exit 1
    fi
    
    # Save database credentials
    cat > api/.env << EOF
# Database Configuration
DB_HOST=localhost
DB_NAME=$DB_NAME
DB_USER=$DB_USER
DB_PASS=$DB_PASSWORD

# JWT Configuration
JWT_SECRET=$(openssl rand -base64 32)
JWT_ALGORITHM=HS256
JWT_EXPIRY=604800

# API Configuration
API_VERSION=v1

# File Upload Configuration
UPLOAD_PATH=../uploads/
MAX_FILE_SIZE=5242880
ALLOWED_FILE_TYPES=jpg,jpeg,png,pdf,doc,docx

# CORS Configuration
ALLOWED_ORIGINS=http://localhost:3000,https://yourdomain.com
EOF
    
    print_status "Database configuration saved"
}

# Frontend setup
setup_frontend() {
    print_step "Setting up frontend..."
    
    cd frontend || exit 1
    
    # Install dependencies
    print_step "Installing frontend dependencies..."
    npm install
    
    if [ $? -ne 0 ]; then
        print_error "Frontend dependencies installation failed"
        exit 1
    fi
    
    # Build frontend
    print_step "Building frontend..."
    npm run build
    
    if [ $? -ne 0 ]; then
        print_error "Frontend build failed"
        exit 1
    fi
    
    print_status "Frontend built successfully"
    cd ..
}

# Backend setup
setup_backend() {
    print_step "Setting up backend..."
    
    # Create uploads directory
    mkdir -p uploads
    chmod 755 uploads
    
    # Set proper permissions for API
    chmod 755 api/index.php
    chmod 755 api/controllers/
    chmod 755 api/middleware/
    chmod 755 api/config/
    
    # Test PHP configuration
    print_step "Testing PHP configuration..."
    php -v
    
    # Test database connection
    print_step "Testing database connection..."
    cd api && php -r "
        require_once 'config/database.php';
        try {
            \$db = new Database();
            \$conn = \$db->getConnection();
            echo 'Database connection successful\n';
        } catch (Exception \$e) {
            echo 'Database connection failed: ' . \$e->getMessage() . '\n';
            exit(1);
        }
    "
    
    if [ $? -eq 0 ]; then
        print_status "Backend setup successful"
    else
        print_error "Backend setup failed"
        exit 1
    fi
    
    cd ..
}

# Web server configuration
setup_webserver() {
    print_step "Setting up web server configuration..."
    
    # Apache configuration
    cat > apache.conf << 'EOF'
# Madrasa Management System - Apache Configuration
<VirtualHost *:80>
    ServerName yourdomain.com
    DocumentRoot /var/www/html
    
    # Frontend (React App)
    <Directory /var/www/html>
        Options -Indexes +FollowSymLinks
        AllowOverride All
        Require all granted
        
        # Enable URL rewriting
        RewriteEngine On
        
        # Handle React Router
        RewriteCond %{REQUEST_FILENAME} !-f
        RewriteCond %{REQUEST_FILENAME} !-d
        RewriteRule . /index.html [L]
        
        # Security headers
        Header always set X-Content-Type-Options nosniff
        Header always set X-Frame-Options DENY
        Header always set X-XSS-Protection "1; mode=block"
    </Directory>
    
    # API Backend
    Alias /api /var/www/html/api
    <Directory /var/www/html/api>
        Options -Indexes +FollowSymLinks
        AllowOverride All
        Require all granted
        
        # PHP configuration
        <Files "*.php">
            SetHandler "proxy:unix:/run/php/php8.1-fpm.sock|fcgi://localhost"
        </Files>
        
        # Security headers
        Header always set X-Content-Type-Options nosniff
        Header always set X-Frame-Options DENY
    </Directory>
    
    # File uploads
    Alias /uploads /var/www/html/uploads
    <Directory /var/www/html/uploads>
        Options -Indexes +FollowSymLinks
        AllowOverride None
        Require all granted
    </Directory>
    
    # Error and access logs
    ErrorLog ${APACHE_LOG_DIR}/madrasa_error.log
    CustomLog ${APACHE_LOG_DIR}/madrasa_access.log combined
</VirtualHost>
EOF
    
    # Nginx configuration
    cat > nginx.conf << 'EOF'
# Madrasa Management System - Nginx Configuration
server {
    listen 80;
    server_name yourdomain.com;
    root /var/www/html;
    index index.html;
    
    # Frontend (React App)
    location / {
        try_files $uri $uri/ /index.html;
        
        # Cache static assets
        location ~* \.(js|css|png|jpg|jpeg|gif|ico|svg)$ {
            expires 1y;
            add_header Cache-Control "public, immutable";
        }
    }
    
    # API Backend
    location /api/ {
        try_files $uri $uri/ /api/index.php?$query_string;
        
        location ~ ^/api/index\.php$ {
            include fastcgi_params;
            fastcgi_pass unix:/run/php/php8.1-fpm.sock;
            fastcgi_param SCRIPT_FILENAME $document_root$fastcgi_script_name;
            include fastcgi_params;
        }
    }
    
    # File uploads
    location /uploads/ {
        try_files $uri $uri/ =404;
    }
    
    # Security headers
    add_header X-Content-Type-Options nosniff;
    add_header X-Frame-Options DENY;
    add_header X-XSS-Protection "1; mode=block";
    
    # Gzip compression
    gzip on;
    gzip_types text/css application/javascript application/json;
}
EOF
    
    print_status "Web server configuration files created"
    print_warning "Please review and update the domain names in the configuration files"
}

# Create deployment summary
create_summary() {
    print_step "Creating deployment summary..."
    
    cat > DEPLOYMENT_SUMMARY.md << EOF
# Madrasa Management System - Deployment Summary

## ✅ Deployment Completed Successfully!

### System Information
- **Deployment Date**: $(date)
- **Database**: MySQL
- **Backend**: PHP 8+
- **Frontend**: React + TypeScript
- **Web Server**: Apache/Nginx

### Database Credentials
- **Database Name**: $DB_NAME
- **Database User**: $DB_USER
- **Host**: localhost

**Important**: Keep your database credentials secure and update them in production!

### Default Login Credentials
After deployment, you can login with:

**Admin User:**
- Username: admin
- Email: admin@madrasa.com
- Password: password (CHANGE THIS IMMEDIATELY!)

### File Structure
\`\`\`
/var/www/html/
├── index.html          # Frontend (React App)
/api/                  # PHP Backend
│   ├── index.php      # Main API router
│   ├── config/        # Configuration files
│   ├── controllers/   # API Controllers
│   ├── middleware/    # Middleware
│   └── models/        # Data models
├── uploads/           # File uploads
└── assets/            # Static assets
\`\`\`

### Next Steps
1. **Change Default Passwords**: Update admin passwords
2. **SSL Certificate**: Install SSL certificate for HTTPS
3. **Domain Configuration**: Update domain names in server config
4. **Backup Setup**: Configure automated backups
5. **Monitoring**: Set up error monitoring
6. **PWA Installation**: Configure PWA for mobile app

### API Endpoints
- **Base URL**: https://yourdomain.com/api/
- **Health Check**: https://yourdomain.com/api/health
- **Authentication**: POST /api/auth/login
- **Students**: GET/POST /api/students
- **Classes**: GET/POST /api/classes
- **Attendance**: GET/POST /api/attendance
- **Fees**: GET/POST /api/fees

### Security Checklist
- [ ] Update default passwords
- [ ] Configure SSL certificate
- [ ] Set proper file permissions
- [ ] Configure firewall
- [ ] Enable security headers
- [ ] Setup backup system
- [ ] Configure monitoring

### Support
For any issues:
1. Check error logs: /var/log/apache2/madrasa_error.log
2. Test API: https://yourdomain.com/api/health
3. Check database connection
4. Verify file permissions

### Mobile App (PWA)
The system is configured as a PWA (Progressive Web App):
- Users can install it as a mobile app
- Works offline
- Push notifications supported
- Can be converted to Android APK for Play Store

**Deployment completed successfully! 🎉**
EOF
    
    print_status "Deployment summary created: DEPLOYMENT_SUMMARY.md"
}

# Main execution
main() {
    echo "Madrasa Management System - Complete Deployment"
    echo "==============================================="
    echo
    
    check_requirements
    setup_database
    setup_frontend
    setup_backend
    setup_webserver
    create_summary
    
    echo
    echo "🎉 Deployment completed successfully!"
    echo
    echo "Next steps:"
    echo "1. Copy files to your web server"
    echo "2. Configure web server (Apache/Nginx)"
    echo "3. Update domain names in configuration"
    echo "4. Install SSL certificate"
    echo "5. Test the application"
    echo
    echo "For detailed instructions, see DEPLOYMENT_SUMMARY.md"
    echo
    echo "Happy managing your madrasa! 🕌"
}

# Run main function
main "$@"