<?php
/**
 * Main API Router
 * Madrasa Management System
 */

// Set CORS headers
setCorsHeaders();

// Load required files
require_once 'config/database.php';
require_once 'middleware/AuthMiddleware.php';
require_once 'controllers/AuthController.php';

// Basic API Response
class APIResponse {
    public static function success($data = null, $message = 'Success', $code = 200) {
        http_response_code($code);
        header('Content-Type: application/json; charset=utf-8');
        echo json_encode([
            'success' => true,
            'message' => $message,
            'data' => $data,
            'timestamp' => date('c'),
            'api_version' => Config::$api_version
        ], JSON_UNESCAPED_UNICODE);
        exit;
    }

    public static function error($message = 'Error', $code = 400) {
        http_response_code($code);
        header('Content-Type: application/json; charset=utf-8');
        echo json_encode([
            'success' => false,
            'message' => $message,
            'timestamp' => date('c'),
            'api_version' => Config::$api_version
        ], JSON_UNESCAPED_UNICODE);
        exit;
    }
}

// Main Router
class Router {
    private $routes = [];
    private $authController;
    
    public function __construct() {
        $this->authController = new AuthController();
    }

    public function addRoute($method, $path, $callback, $auth = false, $role = null) {
        $this->routes[] = [
            'method' => strtoupper($method),
            'path' => $path,
            'callback' => $callback,
            'auth' => $auth,
            'role' => $role
        ];
    }

    public function dispatch() {
        $requestMethod = $_SERVER['REQUEST_METHOD'];
        $requestUri = parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH);
        $path = str_replace('/api/', '', $requestUri);

        // Remove query parameters
        $path = strtok($path, '?');

        // Handle routes
        foreach ($this->routes as $route) {
            if ($route['method'] === $requestMethod && $this->matchRoute($route['path'], $path)) {
                // Authentication check
                if ($route['auth']) {
                    $user = AuthMiddleware::verifyToken($request);
                    
                    // Role check
                    if ($route['role'] && !AuthMiddleware::checkRole($user, $route['role'])) {
                        AuthMiddleware::forbidden('Insufficient permissions');
                        return;
                    }
                    
                    $_REQUEST['user'] = $user;
                }

                // Execute callback
                $params = $this->getRouteParams($route['path'], $path);
                $request = array_merge($_REQUEST, $params);
                
                call_user_func($route['callback'], $request);
                return;
            }
        }

        // Route not found
        AuthMiddleware::notFound('API endpoint not found');
    }

    private function matchRoute($routePath, $requestPath) {
        $routeParts = explode('/', trim($routePath, '/'));
        $requestParts = explode('/', trim($requestPath, '/'));

        if (count($routeParts) !== count($requestParts)) {
            return false;
        }

        foreach ($routeParts as $index => $part) {
            if (strpos($part, ':') === 0) {
                // Parameter found, continue
                continue;
            } elseif ($part !== $requestParts[$index]) {
                return false;
            }
        }

        return true;
    }

    private function getRouteParams($routePath, $requestPath) {
        $params = [];
        $routeParts = explode('/', trim($routePath, '/'));
        $requestParts = explode('/', trim($requestPath, '/'));

        foreach ($routeParts as $index => $part) {
            if (strpos($part, ':') === 0) {
                $paramName = substr($part, 1);
                $params[$paramName] = $requestParts[$index];
            }
        }

        return $params;
    }
}

// Initialize router
$router = new Router();

// Authentication routes (no auth required for login/register)
$router->addRoute('POST', 'auth/login', [$authController, 'login'], false);
$router->addRoute('POST', 'auth/register', [$authController, 'register'], false);

// Protected routes (auth required)
$router->addRoute('GET', 'auth/profile', [$authController, 'profile'], true);
$router->addRoute('PUT', 'auth/profile', [$authController, 'updateProfile'], true);
$router->addRoute('PUT', 'auth/change-password', [$authController, 'changePassword'], true);
$router->addRoute('POST', 'auth/logout', [$authController, 'logout'], true);

// Dashboard routes
$router->addRoute('GET', 'dashboard', function($request) {
    $user = $request['user'];
    $madrasa_id = $user['madrasa_id'];
    
    $db = new Database();
    $conn = $db->getConnection();
    
    // Get dashboard statistics
    $stats = [
        'total_students' => getCount($conn, "SELECT COUNT(*) FROM students WHERE madrasa_id = ? AND status = 'active'", [$madrasa_id]),
        'total_teachers' => getCount($conn, "SELECT COUNT(*) FROM teachers WHERE madrasa_id = ? AND status = 'active'", [$madrasa_id]),
        'total_classes' => getCount($conn, "SELECT COUNT(*) FROM classes WHERE madrasa_id = ? AND status = 'active'", [$madrasa_id]),
        'today_attendance' => getCount($conn, "SELECT COUNT(*) FROM attendance WHERE madrasa_id = ? AND attendance_date = CURDATE()", [$madrasa_id]),
        'pending_fees' => getCount($conn, "SELECT COUNT(*) FROM fees WHERE madrasa_id = ? AND status IN ('pending', 'partial')", [$madrasa_id]),
        'total_pending_fees' => getSum($conn, "SELECT COALESCE(SUM(amount - paid_amount), 0) FROM fees WHERE madrasa_id = ? AND status IN ('pending', 'partial')", [$madrasa_id])
    ];

    // Get recent activities
    $stmt = $conn->prepare("
        SELECT a.*, u.full_name as user_name 
        FROM activity_logs a
        LEFT JOIN users u ON a.user_id = u.id
        WHERE a.madrasa_id = ?
        ORDER BY a.created_at DESC 
        LIMIT 10
    ");
    $stmt->execute([$madrasa_id]);
    $recent_activities = $stmt->fetchAll();

    APIResponse::success([
        'stats' => $stats,
        'recent_activities' => $recent_activities
    ], 'Dashboard data retrieved');
}, true);

// Students routes
$router->addRoute('GET', 'students', function($request) {
    $user = $request['user'];
    $madrasa_id = $user['madrasa_id'];
    $page = intval($_GET['page'] ?? 1);
    $limit = intval($_GET['limit'] ?? 20);
    $search = $_GET['search'] ?? '';
    
    $db = new Database();
    $conn = $db->getConnection();
    
    $whereClause = "s.madrasa_id = ?";
    $params = [$madrasa_id];
    
    if (!empty($search)) {
        $whereClause .= " AND (s.first_name LIKE ? OR s.last_name LIKE ? OR s.student_id LIKE ? OR s.father_name LIKE ?)";
        $searchTerm = "%$search%";
        $params = array_merge($params, [$searchTerm, $searchTerm, $searchTerm, $searchTerm]);
    }
    
    // Get total count
    $countStmt = $conn->prepare("SELECT COUNT(*) FROM students s WHERE $whereClause");
    $countStmt->execute($params);
    $total = $countStmt->fetchColumn();
    
    // Get students
    $offset = ($page - 1) * $limit;
    $stmt = $conn->prepare("
        SELECT s.*, c.class_name, c.grade_level, u.email as user_email
        FROM students s
        LEFT JOIN classes c ON s.current_class_id = c.id
        LEFT JOIN users u ON s.user_id = u.id
        WHERE $whereClause
        ORDER BY s.first_name, s.last_name
        LIMIT ? OFFSET ?
    ");
    $params[] = $limit;
    $params[] = $offset;
    $stmt->execute($params);
    $students = $stmt->fetchAll();
    
    APIResponse::success([
        'students' => $students,
        'pagination' => [
            'current_page' => $page,
            'per_page' => $limit,
            'total' => $total,
            'total_pages' => ceil($total / $limit)
        ]
    ]);
}, true, 'admin');

$router->addRoute('POST', 'students', function($request) {
    $user = $request['user'];
    $data = json_decode(file_get_contents('php://input'), true);
    
    $db = new Database();
    $conn = $db->getConnection();
    
    // Create user account
    $username = 'student_' . time();
    $password = password_hash('123456', PASSWORD_DEFAULT);
    
    $userStmt = $conn->prepare("
        INSERT INTO users (username, email, password_hash, full_name, role) 
        VALUES (?, ?, ?, ?, 'student')
    ");
    $fullName = $data['first_name'] . ' ' . $data['last_name'];
    $userStmt->execute([$username, $data['email'], $password, $fullName]);
    $user_id = $conn->lastInsertId();
    
    // Generate student ID
    $studentId = 'S' . date('Y') . str_pad($user_id, 4, '0', STR_PAD_LEFT);
    
    // Create student record
    $stmt = $conn->prepare("
        INSERT INTO students (user_id, madrasa_id, student_id, first_name, last_name, father_name, 
                             mother_name, date_of_birth, gender, phone, guardian_phone, admission_date)
        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
    ");
    
    $stmt->execute([
        $user_id, $user['madrasa_id'], $studentId,
        $data['first_name'], $data['last_name'], $data['father_name'],
        $data['mother_name'] ?? null, $data['date_of_birth'] ?? null,
        $data['gender'], $data['phone'] ?? null, $data['guardian_phone'] ?? null,
        $data['admission_date'] ?? date('Y-m-d')
    ]);
    
    APIResponse::created(['student_id' => $studentId], 'Student created successfully');
}, true, 'admin');

$router->addRoute('GET', 'students/:id', function($request) {
    $studentId = $request['id'];
    $user = $request['user'];
    
    $db = new Database();
    $conn = $db->getConnection();
    
    $stmt = $conn->prepare("
        SELECT s.*, c.class_name, c.grade_level, u.email as user_email
        FROM students s
        LEFT JOIN classes c ON s.current_class_id = c.id
        LEFT JOIN users u ON s.user_id = u.id
        WHERE s.id = ? AND s.madrasa_id = ?
    ");
    $stmt->execute([$studentId, $user['madrasa_id']]);
    $student = $stmt->fetch();
    
    if (!$student) {
        AuthMiddleware::notFound('Student not found');
    }
    
    APIResponse::success($student);
}, true, 'admin');

$router->addRoute('PUT', 'students/:id', function($request) {
    $studentId = $request['id'];
    $user = $request['user'];
    $data = json_decode(file_get_contents('php://input'), true);
    
    $db = new Database();
    $conn = $db->getConnection();
    
    $updateFields = [];
    $params = [];
    
    $allowedFields = ['first_name', 'last_name', 'father_name', 'mother_name', 'date_of_birth', 
                     'gender', 'phone', 'guardian_phone', 'current_class_id', 'status'];
    
    foreach ($allowedFields as $field) {
        if (isset($data[$field])) {
            $updateFields[] = "$field = ?";
            $params[] = $data[$field];
        }
    }
    
    if (empty($updateFields)) {
        AuthMiddleware::badRequest('No valid fields to update');
    }
    
    $params[] = $studentId;
    $params[] = $user['madrasa_id'];
    
    $stmt = $conn->prepare("UPDATE students SET " . implode(', ', $updateFields) . " WHERE id = ? AND madrasa_id = ?");
    $stmt->execute($params);
    
    APIResponse::success(null, 'Student updated successfully');
}, true, 'admin');

$router->addRoute('DELETE', 'students/:id', function($request) {
    $studentId = $request['id'];
    $user = $request['user'];
    
    $db = new Database();
    $conn = $db->getConnection();
    
    $stmt = $conn->prepare("DELETE FROM students WHERE id = ? AND madrasa_id = ?");
    $stmt->execute([$studentId, $user['madrasa_id']]);
    
    APIResponse::success(null, 'Student deleted successfully');
}, true, 'admin');

// Classes routes
$router->addRoute('GET', 'classes', function($request) {
    $user = $request['user'];
    $madrasa_id = $user['madrasa_id'];
    
    $db = new Database();
    $conn = $db->getConnection();
    
    $stmt = $conn->prepare("
        SELECT c.*, t.first_name, t.last_name, t.teacher_id
        FROM classes c
        LEFT JOIN teachers t ON c.teacher_id = t.id
        WHERE c.madrasa_id = ?
        ORDER BY c.grade_level, c.class_name
    ");
    $stmt->execute([$madrasa_id]);
    $classes = $stmt->fetchAll();
    
    APIResponse::success($classes);
}, true, 'admin');

$router->addRoute('POST', 'classes', function($request) {
    $user = $request['user'];
    $data = json_decode(file_get_contents('php://input'), true);
    
    $db = new Database();
    $conn = $db->getConnection();
    
    $stmt = $conn->prepare("
        INSERT INTO classes (madrasa_id, class_name, grade_level, section, teacher_id, class_time, room_number)
        VALUES (?, ?, ?, ?, ?, ?, ?)
    ");
    $stmt->execute([
        $user['madrasa_id'], $data['class_name'], $data['grade_level'],
        $data['section'] ?? null, $data['teacher_id'] ?? null,
        $data['class_time'] ?? null, $data['room_number'] ?? null
    ]);
    
    APIResponse::created(['class_id' => $conn->lastInsertId()], 'Class created successfully');
}, true, 'admin');

// Attendance routes
$router->addRoute('GET', 'attendance', function($request) {
    $user = $request['user'];
    $date = $_GET['date'] ?? date('Y-m-d');
    $class_id = $_GET['class_id'] ?? null;
    
    $db = new Database();
    $conn = $db->getConnection();
    
    $whereClause = "a.madrasa_id = ? AND a.attendance_date = ?";
    $params = [$user['madrasa_id'], $date];
    
    if ($class_id) {
        $whereClause .= " AND a.class_id = ?";
        $params[] = $class_id;
    }
    
    $stmt = $conn->prepare("
        SELECT a.*, s.first_name, s.last_name, s.student_id, c.class_name
        FROM attendance a
        LEFT JOIN students s ON a.student_id = s.id
        LEFT JOIN classes c ON a.class_id = c.id
        WHERE $whereClause
        ORDER BY s.first_name, s.last_name
    ");
    $stmt->execute($params);
    $attendance = $stmt->fetchAll();
    
    APIResponse::success($attendance);
}, true, 'teacher');

$router->addRoute('POST', 'attendance', function($request) {
    $user = $request['user'];
    $data = json_decode(file_get_contents('php://input'), true);
    
    $db = new Database();
    $conn = $db->getConnection();
    
    $stmt = $conn->prepare("
        INSERT INTO attendance (student_id, class_id, teacher_id, madrasa_id, attendance_date, status, remarks, marked_by)
        VALUES (?, ?, ?, ?, ?, ?, ?, ?)
    ");
    
    foreach ($data['records'] as $record) {
        $stmt->execute([
            $record['student_id'], $record['class_id'] ?? null, 
            $user['user_id'] ?? $user['id'], $user['madrasa_id'],
            $data['date'], $record['status'], $record['remarks'] ?? null, $user['id']
        ]);
    }
    
    APIResponse::created(null, 'Attendance marked successfully');
}, true, 'teacher');

// Fees routes
$router->addRoute('GET', 'fees', function($request) {
    $user = $request['user'];
    $student_id = $_GET['student_id'] ?? null;
    $status = $_GET['status'] ?? null;
    
    $db = new Database();
    $conn = $db->getConnection();
    
    $whereClause = "f.madrasa_id = ?";
    $params = [$user['madrasa_id']];
    
    if ($student_id) {
        $whereClause .= " AND f.student_id = ?";
        $params[] = $student_id;
    }
    
    if ($status) {
        $whereClause .= " AND f.status = ?";
        $params[] = $status;
    }
    
    $stmt = $conn->prepare("
        SELECT f.*, s.first_name, s.last_name, s.student_id
        FROM fees f
        LEFT JOIN students s ON f.student_id = s.id
        WHERE $whereClause
        ORDER BY f.due_date DESC
    ");
    $stmt->execute($params);
    $fees = $stmt->fetchAll();
    
    APIResponse::success($fees);
}, true);

$router->addRoute('POST', 'fees', function($request) {
    $user = $request['user'];
    $data = json_decode(file_get_contents('php://input'), true);
    
    $db = new Database();
    $conn = $db->getConnection();
    
    $stmt = $conn->prepare("
        INSERT INTO fees (student_id, madrasa_id, fee_type, amount, due_date, status)
        VALUES (?, ?, ?, ?, ?, ?)
    ");
    $stmt->execute([
        $data['student_id'], $user['madrasa_id'], $data['fee_type'],
        $data['amount'], $data['due_date'], $data['status'] ?? 'pending'
    ]);
    
    APIResponse::created(['fee_id' => $conn->lastInsertId()], 'Fee record created');
}, true, 'admin');

// Reports routes
$router->addRoute('GET', 'reports/students', function($request) {
    $user = $request['user'];
    $class_id = $_GET['class_id'] ?? null;
    
    $db = new Database();
    $conn = $db->getConnection();
    
    $whereClause = "s.madrasa_id = ?";
    $params = [$user['madrasa_id']];
    
    if ($class_id) {
        $whereClause .= " AND s.current_class_id = ?";
        $params[] = $class_id;
    }
    
    $stmt = $conn->prepare("
        SELECT 
            s.*, 
            c.class_name,
            COUNT(DISTINCT a.id) as total_days,
            SUM(CASE WHEN a.status = 'present' THEN 1 ELSE 0 END) as present_days,
            ROUND((SUM(CASE WHEN a.status = 'present' THEN 1 ELSE 0 END) / COUNT(DISTINCT a.id)) * 100, 2) as attendance_percentage
        FROM students s
        LEFT JOIN classes c ON s.current_class_id = c.id
        LEFT JOIN attendance a ON s.id = a.student_id
        WHERE $whereClause
        GROUP BY s.id
        ORDER BY s.first_name, s.last_name
    ");
    $stmt->execute($params);
    $students = $stmt->fetchAll();
    
    APIResponse::success($students);
}, true, 'admin');

$router->addRoute('GET', 'reports/financial', function($request) {
    $user = $request['user'];
    $start_date = $_GET['start_date'] ?? date('Y-m-01');
    $end_date = $_GET['end_date'] ?? date('Y-m-t');
    
    $db = new Database();
    $conn = $db->getConnection();
    
    // Get income
    $stmt = $conn->prepare("
        SELECT 
            DATE_FORMAT(received_date, '%Y-%m') as month,
            source,
            SUM(amount) as total_amount,
            COUNT(*) as transactions
        FROM income 
        WHERE madrasa_id = ? AND received_date BETWEEN ? AND ?
        GROUP BY DATE_FORMAT(received_date, '%Y-%m'), source
        ORDER BY month DESC
    ");
    $stmt->execute([$user['madrasa_id'], $start_date, $end_date]);
    $income = $stmt->fetchAll();
    
    // Get expenses
    $stmt = $conn->prepare("
        SELECT 
            DATE_FORMAT(expense_date, '%Y-%m') as month,
            category,
            SUM(amount) as total_amount,
            COUNT(*) as transactions
        FROM expenses 
        WHERE madrasa_id = ? AND expense_date BETWEEN ? AND ?
        GROUP BY DATE_FORMAT(expense_date, '%Y-%m'), category
        ORDER BY month DESC
    ");
    $stmt->execute([$user['madrasa_id'], $start_date, $end_date]);
    $expenses = $stmt->fetchAll();
    
    APIResponse::success([
        'income' => $income,
        'expenses' => $expenses,
        'period' => ['start' => $start_date, 'end' => $end_date]
    ]);
}, true, 'admin');

// Fallback 404 handler
$router->addRoute('GET', 'health', function() {
    APIResponse::success([
        'status' => 'healthy',
        'version' => Config::$api_version,
        'timestamp' => date('c')
    ]);
}, false);

// Dispatch the request
$router->dispatch();

/**
 * Helper functions
 */
function getCount($conn, $query, $params = []) {
    $stmt = $conn->prepare($query);
    $stmt->execute($params);
    return $stmt->fetchColumn();
}

function getSum($conn, $query, $params = []) {
    $stmt = $conn->prepare($query);
    $stmt->execute($params);
    return floatval($stmt->fetchColumn());
}
?>