<?php
/**
 * Authentication Controller
 * Madrasa Management System
 */

class AuthController {
    private $db;
    
    public function __construct() {
        $this->db = new Database();
    }

    /**
     * User Login
     * POST /api/auth/login
     */
    public function login() {
        try {
            $data = json_decode(file_get_contents('php://input'), true);
            
            if (!isset($data['username']) || !isset($data['password'])) {
                AuthMiddleware::badRequest('Username and password are required');
            }

            $username = trim($data['username']);
            $password = $data['password'];

            $conn = $this->db->getConnection();
            $stmt = $conn->prepare("
                SELECT u.id, u.username, u.email, u.password_hash, u.full_name, u.role, u.status,
                       t.madrasa_id as teacher_madrasa_id, s.madrasa_id as student_madrasa_id
                FROM users u
                LEFT JOIN teachers t ON u.id = t.user_id
                LEFT JOIN students s ON u.id = s.user_id
                WHERE u.username = ? AND u.status = 'active'
            ");
            $stmt->execute([$username]);
            $user = $stmt->fetch();

            if (!$user || !password_verify($password, $user['password_hash'])) {
                AuthMiddleware::unauthorized('Invalid credentials');
            }

            // Update last login
            $updateStmt = $conn->prepare("UPDATE users SET last_login = NOW() WHERE id = ?");
            $updateStmt->execute([$user['id']]);

            // Determine madrasa access
            $madrasa_id = $user['role'] === 'teacher' ? $user['teacher_madrasa_id'] : 
                          ($user['role'] === 'student' ? $user['student_madrasa_id'] : 1);

            $payload = [
                'user_id' => $user['id'],
                'username' => $user['username'],
                'email' => $user['email'],
                'role' => $user['role'],
                'full_name' => $user['full_name'],
                'madrasa_id' => $madrasa_id,
                'iat' => time(),
                'exp' => time() + Config::$jwt_expiry
            ];

            $token = JWT::encode($payload);

            // Get additional user data based on role
            $userData = $this->getUserDataByRole($user, $madrasa_id);

            AuthMiddleware::ok([
                'user' => [
                    'id' => $user['id'],
                    'username' => $user['username'],
                    'email' => $user['email'],
                    'full_name' => $user['full_name'],
                    'role' => $user['role'],
                    'madrasa_id' => $madrasa_id
                ],
                'token' => $token,
                'additional_data' => $userData
            ], 'Login successful');

        } catch (Exception $e) {
            error_log("Login error: " . $e->getMessage());
            AuthMiddleware::serverError('Login failed');
        }
    }

    /**
     * User Registration
     * POST /api/auth/register
     */
    public function register() {
        try {
            $data = json_decode(file_get_contents('php://input'), true);
            
            // Validation
            $required = ['username', 'email', 'password', 'full_name', 'role'];
            foreach ($required as $field) {
                if (empty($data[$field])) {
                    AuthMiddleware::badRequest("$field is required");
                }
            }

            if (!in_array($data['role'], ['admin', 'teacher', 'student'])) {
                AuthMiddleware::badRequest('Invalid role');
            }

            $username = trim($data['username']);
            $email = trim($data['email']);
            $password = $data['password'];
            $full_name = trim($data['full_name']);
            $role = $data['role'];
            $madrasa_id = $data['madrasa_id'] ?? 1;

            // Validate password strength
            if (strlen($password) < 6) {
                AuthMiddleware::badRequest('Password must be at least 6 characters long');
            }

            $conn = $this->db->getConnection();

            // Check if username or email already exists
            $checkStmt = $conn->prepare("SELECT id FROM users WHERE username = ? OR email = ?");
            $checkStmt->execute([$username, $email]);
            if ($checkStmt->fetch()) {
                AuthMiddleware::badRequest('Username or email already exists');
            }

            $password_hash = password_hash($password, PASSWORD_DEFAULT);

            $this->db->beginTransaction();

            try {
                // Create user
                $userStmt = $conn->prepare("
                    INSERT INTO users (username, email, password_hash, full_name, role) 
                    VALUES (?, ?, ?, ?, ?)
                ");
                $userStmt->execute([$username, $email, $password_hash, $full_name, $role]);
                $user_id = $conn->lastInsertId();

                // Create role-specific record
                $this->createRoleSpecificRecord($conn, $user_id, $role, $madrasa_id, $data);

                $this->db->commit();

                // Generate token for automatic login
                $payload = [
                    'user_id' => $user_id,
                    'username' => $username,
                    'email' => $email,
                    'role' => $role,
                    'full_name' => $full_name,
                    'madrasa_id' => $madrasa_id,
                    'iat' => time(),
                    'exp' => time() + Config::$jwt_expiry
                ];
                $token = JWT::encode($payload);

                AuthMiddleware::created([
                    'user_id' => $user_id,
                    'username' => $username,
                    'token' => $token
                ], 'Registration successful');

            } catch (Exception $e) {
                $this->db->rollback();
                throw $e;
            }

        } catch (Exception $e) {
            error_log("Registration error: " . $e->getMessage());
            AuthMiddleware::serverError('Registration failed');
        }
    }

    /**
     * Get Current User Profile
     * GET /api/auth/profile
     */
    public function profile() {
        try {
            $user = AuthMiddleware::verifyToken($request);
            $userData = $this->getCompleteUserData($user);

            AuthMiddleware::ok($userData);

        } catch (Exception $e) {
            error_log("Profile error: " . $e->getMessage());
            AuthMiddleware::serverError('Failed to fetch profile');
        }
    }

    /**
     * Update User Profile
     * PUT /api/auth/profile
     */
    public function updateProfile() {
        try {
            $user = AuthMiddleware::verifyToken($_REQUEST);
            $data = json_decode(file_get_contents('php://input'), true);

            $conn = $this->db->getConnection();
            $updateFields = [];
            $params = [];

            $allowedFields = ['full_name', 'phone', 'email'];
            foreach ($allowedFields as $field) {
                if (isset($data[$field])) {
                    $updateFields[] = "$field = ?";
                    $params[] = $data[$field];
                }
            }

            if (empty($updateFields)) {
                AuthMiddleware::badRequest('No valid fields to update');
            }

            $params[] = $user['id'];
            
            $stmt = $conn->prepare("UPDATE users SET " . implode(', ', $updateFields) . " WHERE id = ?");
            $stmt->execute($params);

            AuthMiddleware::ok(null, 'Profile updated successfully');

        } catch (Exception $e) {
            error_log("Update profile error: " . $e->getMessage());
            AuthMiddleware::serverError('Failed to update profile');
        }
    }

    /**
     * Change Password
     * PUT /api/auth/change-password
     */
    public function changePassword() {
        try {
            $user = AuthMiddleware::verifyToken($_REQUEST);
            $data = json_decode(file_get_contents('php://input'), true);

            if (!isset($data['current_password']) || !isset($data['new_password'])) {
                AuthMiddleware::badRequest('Current password and new password are required');
            }

            $currentPassword = $data['current_password'];
            $newPassword = $data['new_password'];

            if (strlen($newPassword) < 6) {
                AuthMiddleware::badRequest('New password must be at least 6 characters long');
            }

            $conn = $this->db->getConnection();
            
            // Verify current password
            $stmt = $conn->prepare("SELECT password_hash FROM users WHERE id = ?");
            $stmt->execute([$user['id']]);
            $userData = $stmt->fetch();

            if (!password_verify($currentPassword, $userData['password_hash'])) {
                AuthMiddleware::badRequest('Current password is incorrect');
            }

            // Update password
            $newHash = password_hash($newPassword, PASSWORD_DEFAULT);
            $updateStmt = $conn->prepare("UPDATE users SET password_hash = ? WHERE id = ?");
            $updateStmt->execute([$newHash, $user['id']]);

            AuthMiddleware::ok(null, 'Password changed successfully');

        } catch (Exception $e) {
            error_log("Change password error: " . $e->getMessage());
            AuthMiddleware::serverError('Failed to change password');
        }
    }

    /**
     * Logout (Client-side token invalidation)
     * POST /api/auth/logout
     */
    public function logout() {
        // In a stateless JWT system, logout is typically handled client-side
        // by removing the token. We can log the logout event here.
        logRequest($_REQUEST['user']['id'] ?? null, 'logout', 'User logged out');
        
        AuthMiddleware::ok(null, 'Logged out successfully');
    }

    /**
     * Create role-specific record
     */
    private function createRoleSpecificRecord($conn, $user_id, $role, $madrasa_id, $data) {
        switch ($role) {
            case 'teacher':
                $teacherId = 'T' . date('Y') . str_pad($user_id, 4, '0', STR_PAD_LEFT);
                $stmt = $conn->prepare("
                    INSERT INTO teachers (user_id, madrasa_id, teacher_id, first_name, last_name, hire_date) 
                    VALUES (?, ?, ?, ?, ?, ?)
                ");
                $names = explode(' ', $data['full_name'], 2);
                $stmt->execute([
                    $user_id, $madrasa_id, $teacherId, 
                    $names[0] ?? '', $names[1] ?? '', date('Y-m-d')
                ]);
                break;

            case 'student':
                $studentId = 'S' . date('Y') . str_pad($user_id, 4, '0', STR_PAD_LEFT);
                $stmt = $conn->prepare("
                    INSERT INTO students (user_id, madrasa_id, student_id, first_name, last_name, father_name, admission_date) 
                    VALUES (?, ?, ?, ?, ?, ?, ?)
                ");
                $names = explode(' ', $data['full_name'], 2);
                $stmt->execute([
                    $user_id, $madrasa_id, $studentId,
                    $names[0] ?? '', $names[1] ?? '', 
                    $data['father_name'] ?? '', date('Y-m-d')
                ]);
                break;

            case 'admin':
                // Admin doesn't need a separate record
                break;
        }
    }

    /**
     * Get user data by role
     */
    private function getUserDataByRole($user, $madrasa_id) {
        if ($user['role'] === 'teacher') {
            return $this->getTeacherData($user['id']);
        } elseif ($user['role'] === 'student') {
            return $this->getStudentData($user['id']);
        }
        return null;
    }

    /**
     * Get teacher data
     */
    private function getTeacherData($user_id) {
        $conn = $this->db->getConnection();
        $stmt = $conn->prepare("
            SELECT t.*, c.class_name, c.grade_level 
            FROM teachers t 
            LEFT JOIN classes c ON t.id = c.teacher_id 
            WHERE t.user_id = ?
        ");
        $stmt->execute([$user_id]);
        return $stmt->fetch();
    }

    /**
     * Get student data
     */
    private function getStudentData($user_id) {
        $conn = $this->db->getConnection();
        $stmt = $conn->prepare("
            SELECT s.*, c.class_name, c.grade_level,
                   COUNT(DISTINCT a.id) as total_attendance
            FROM students s 
            LEFT JOIN classes c ON s.current_class_id = c.id
            LEFT JOIN attendance a ON s.id = a.student_id
            WHERE s.user_id = ?
            GROUP BY s.id
        ");
        $stmt->execute([$user_id]);
        return $stmt->fetch();
    }

    /**
     * Get complete user data
     */
    private function getCompleteUserData($user) {
        $conn = $this->db->getConnection();
        
        // Get basic user data
        $stmt = $conn->prepare("
            SELECT u.id, u.username, u.email, u.full_name, u.phone, u.role, u.avatar_url, u.created_at
            FROM users u WHERE u.id = ?
        ");
        $stmt->execute([$user['id']]);
        $userData = $stmt->fetch();

        // Get role-specific data
        if ($user['role'] === 'teacher') {
            $stmt = $conn->prepare("
                SELECT t.teacher_id, t.qualification, t.experience_years, t.subject_specialization,
                       COUNT(DISTINCT c.id) as assigned_classes
                FROM teachers t
                LEFT JOIN classes c ON t.id = c.teacher_id
                WHERE t.user_id = ?
                GROUP BY t.id
            ");
            $stmt->execute([$user['id']]);
            $roleData = $stmt->fetch();
        } elseif ($user['role'] === 'student') {
            $stmt = $conn->prepare("
                SELECT s.student_id, s.father_name, s.gender, s.current_class_id,
                       c.class_name, c.grade_level,
                       COUNT(DISTINCT a.id) as total_attendance
                FROM students s
                LEFT JOIN classes c ON s.current_class_id = c.id
                LEFT JOIN attendance a ON s.id = a.student_id
                WHERE s.user_id = ?
                GROUP BY s.id
            ");
            $stmt->execute([$user['id']]);
            $roleData = $stmt->fetch();
        }

        return array_merge($userData, $roleData ?? []);
    }
}
?>