# 🔧 Complete Installation Guide
## Madrasa Management System v2.0

### 📋 **System Requirements**

#### **Server Requirements:**
- **PHP**: 8.0 or higher
- **MySQL**: 5.7 or higher (or MariaDB 10.3+)
- **Node.js**: 16+ (for building frontend)
- **Web Server**: Apache 2.4+ or Nginx 1.18+
- **SSL Certificate**: Recommended for production

#### **Server Resources:**
- **RAM**: Minimum 2GB (4GB recommended)
- **Storage**: Minimum 5GB free space
- **Bandwidth**: 100GB/month (for moderate usage)

### 🚀 **Step-by-Step Installation**

#### **Step 1: Server Preparation**

```bash
# Update system packages
sudo apt update && sudo apt upgrade -y

# Install required packages
sudo apt install -y php8.1 php8.1-fpm php8.1-mysql php8.1-json php8.1-curl php8.1-mbstring php8.1-xml php8.1-zip php8.1-gd php8.1-intl mysql-server nginx nodejs npm

# Install Composer (PHP dependency manager)
curl -sS https://getcomposer.org/installer | php
sudo mv composer.phar /usr/local/bin/composer

# Install Node.js dependencies
sudo npm install -g serve
```

#### **Step 2: Database Setup**

```sql
-- Login to MySQL
mysql -u root -p

-- Create database and user
CREATE DATABASE madrasa_management CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
CREATE USER 'madrasa_user'@'localhost' IDENTIFIED BY 'secure_password_here';
GRANT ALL PRIVILEGES ON madrasa_management.* TO 'madrasa_user'@'localhost';
FLUSH PRIVILEGES;
EXIT;

-- Import database schema
mysql -u madrasa_user -p madrasa_management < database/schema.sql
```

#### **Step 3: Upload Files**

**Option A: Using cPanel File Manager**
1. Login to your cPanel
2. Open File Manager
3. Upload all files to public_html folder
4. Extract files if uploaded as ZIP

**Option B: Using FTP/SFTP**
```bash
# Upload files to your server
# Frontend files go to /var/www/html/
# Backend files go to /var/www/html/api/
```

**Option C: Direct Server Upload**
```bash
# Copy files to web directory
sudo cp -r madrasa-redesigned/frontend/dist/* /var/www/html/
sudo cp -r madrasa-redesigned/api/ /var/www/html/
sudo cp -r madrasa-redesigned/uploads/ /var/www/html/
```

#### **Step 4: Configuration**

**Backend Configuration:**
```php
// Edit api/config/database.php
private $host = 'localhost';
private $db_name = 'madrasa_management';
private $username = 'madrasa_user';
private $password = 'your_secure_password';
```

**Or create .env file in api/ folder:**
```env
DB_HOST=localhost
DB_NAME=madrasa_management
DB_USER=madrasa_user
DB_PASS=your_secure_password
JWT_SECRET=your-super-secret-jwt-key
```

**Frontend Configuration:**
```bash
# Edit frontend/.env
VITE_API_URL=https://yourdomain.com/api
```

#### **Step 5: File Permissions**

```bash
# Set proper permissions
sudo chown -R www-data:www-data /var/www/html/
sudo chmod -R 755 /var/www/html/
sudo chmod -R 777 /var/www/html/uploads/
sudo chmod 644 /var/www/html/api/index.php
```

#### **Step 6: Web Server Configuration**

**Apache Configuration (Recommended):**

```apache
# /etc/apache2/sites-available/madrasa.conf
<VirtualHost *:80>
    ServerName yourdomain.com
    DocumentRoot /var/www/html
    
    <Directory /var/www/html>
        Options -Indexes +FollowSymLinks
        AllowOverride All
        Require all granted
        
        # Handle React Router
        RewriteEngine On
        RewriteCond %{REQUEST_FILENAME} !-f
        RewriteCond %{REQUEST_FILENAME} !-d
        RewriteRule . /index.html [L]
    </Directory>
    
    # API Backend
    Alias /api /var/www/html/api
    <Directory /var/www/html/api>
        Options -Indexes +FollowSymLinks
        AllowOverride All
        Require all granted
    </Directory>
    
    # File uploads
    Alias /uploads /var/www/html/uploads
    <Directory /var/www/html/uploads>
        Options -Indexes +FollowSymLinks
        AllowOverride None
        Require all granted
    </Directory>
</VirtualHost>
```

**Enable Apache modules and site:**
```bash
sudo a2enmod rewrite
sudo a2enmod headers
sudo a2ensite madrasa.conf
sudo systemctl restart apache2
```

**Nginx Configuration:**

```nginx
# /etc/nginx/sites-available/madrasa
server {
    listen 80;
    server_name yourdomain.com;
    root /var/www/html;
    index index.html;
    
    # Frontend
    location / {
        try_files $uri $uri/ /index.html;
    }
    
    # API Backend
    location /api/ {
        try_files $uri $uri/ /api/index.php?$query_string;
    }
    
    # File uploads
    location /uploads/ {
        try_files $uri $uri/ =404;
    }
}
```

**Enable Nginx site:**
```bash
sudo ln -s /etc/nginx/sites-available/madrasa /etc/nginx/sites-enabled/
sudo nginx -t
sudo systemctl restart nginx
```

#### **Step 7: SSL Certificate (Production)**

**Using Let's Encrypt (Free):**
```bash
# Install Certbot
sudo apt install certbot python3-certbot-apache

# Get SSL certificate
sudo certbot --apache -d yourdomain.com

# Auto-renewal
sudo crontab -e
# Add: 0 12 * * * /usr/bin/certbot renew --quiet
```

#### **Step 8: Test Installation**

1. **Test Database Connection:**
```bash
php api/test_connection.php
```

2. **Test API Endpoint:**
```bash
curl https://yourdomain.com/api/health
```

3. **Test Frontend:**
Open browser and navigate to: https://yourdomain.com

### 🔐 **Security Configuration**

#### **1. Change Default Passwords**
```sql
-- Update default admin password
UPDATE users SET password_hash = '$2y$10$newhashedpassword' WHERE username = 'admin';
```

#### **2. Configure Firewall**
```bash
sudo ufw enable
sudo ufw allow ssh
sudo ufw allow 80
sudo ufw allow 443
```

#### **3. Security Headers**
Add to your web server configuration:
```apache
# Security headers
Header always set X-Content-Type-Options nosniff
Header always set X-Frame-Options DENY
Header always set X-XSS-Protection "1; mode=block"
Header always set Strict-Transport-Security "max-age=31536000; includeSubDomains"
```

### 📱 **Mobile App (PWA) Setup**

#### **PWA Features:**
- Installable on mobile devices
- Offline support
- Push notifications
- Native app experience

#### **Convert to Android APK:**

**Option 1: Using PWA Builder (Microsoft)**
1. Go to https://pwabuilder.com
2. Enter your website URL
3. Click "Package for Stores"
4. Follow instructions for Play Store

**Option 2: Using Bubblewrap (Google)**
```bash
# Install Bubblewrap
npm i -g @bubblewrap/cli

# Initialize PWA
bubblewrap init --manifest https://yourdomain.com/manifest.json

# Build APK
bubblewrap build
```

**Option 3: Manual Conversion**
1. Use tools like "WebAPK" or "PWA Builder"
2. Provide app icons and details
3. Generate APK file

### 💰 **AdSense Integration**

#### **Step 1: AdSense Account**
1. Go to https://www.google.com/adsense/
2. Create account and add your website
3. Wait for approval (1-2 weeks)

#### **Step 2: Add Ads to App**

**In React components:**
```jsx
import { AdSense } from 'react-adsense'

// Banner Ad
<AdSense
  client="ca-pub-your-publisher-id"
  slot="your-slot-id"
  format="auto"
  responsive={true}
/>

// Interstitial Ad
<AdSense
  client="ca-pub-your-publisher-id"
  slot="your-slot-id"
  format="fluid"
  layout="in-article"
/>
```

**Ad Placements:**
- Dashboard banner
- Student list interstitial
- Fee payment confirmation
- Report view bottom

### 🔧 **Troubleshooting**

#### **Common Issues:**

**1. Database Connection Error**
```bash
# Check MySQL status
sudo systemctl status mysql

# Test connection
mysql -u madrasa_user -p madrasa_management
```

**2. API Not Working**
```bash
# Check PHP errors
tail -f /var/log/apache2/error.log
# or
tail -f /var/log/nginx/error.log

# Test PHP
php -v
php -m | grep mysql
```

**3. Frontend Not Loading**
- Check web server configuration
- Verify file permissions
- Clear browser cache
- Check console for errors

**4. CORS Issues**
Update `api/middleware/AuthMiddleware.php`:
```php
public static $allowed_origins = [
    'https://yourdomain.com',
    'https://www.yourdomain.com'
];
```

### 📊 **Performance Optimization**

#### **1. Enable Caching**
```apache
# Apache mod_cache
<IfModule mod_cache.c>
    CacheQuickHandler on
    CacheLock on
    CacheLockPath /tmp/mod_cache_lock
    CacheDefaultExpire 3600
</IfModule>
```

#### **2. Database Optimization**
```sql
-- Add indexes
CREATE INDEX idx_students_name ON students(first_name, last_name);
CREATE INDEX idx_attendance_date ON attendance(attendance_date);
CREATE INDEX idx_fees_status ON fees(status, due_date);
```

#### **3. Image Optimization**
- Compress images before upload
- Use WebP format when possible
- Implement lazy loading

### 📈 **Monitoring & Maintenance**

#### **1. Error Monitoring**
```bash
# Monitor error logs
tail -f /var/log/apache2/madrasa_error.log

# Set up log rotation
sudo nano /etc/logrotate.d/madrasa
```

#### **2. Backup System**
```bash
#!/bin/bash
# backup.sh
DATE=$(date +%Y%m%d_%H%M%S)
mysqldump -u madrasa_user -p madrasa_management > backup_$DATE.sql
tar -czf uploads_backup_$DATE.tar.gz /var/www/html/uploads/
```

#### **3. Automated Updates**
```bash
# Add to crontab
0 2 * * * /path/to/backup.sh
0 3 * * 0 /usr/bin/certbot renew --quiet
```

### 🎯 **Go Live Checklist**

- [ ] Database setup and tested
- [ ] Files uploaded and permissions set
- [ ] Web server configured
- [ ] SSL certificate installed
- [ ] Default passwords changed
- [ ] Security headers configured
- [ ] API endpoints tested
- [ ] Frontend working
- [ ] PWA configured
- [ ] AdSense setup (if applicable)
- [ ] Backup system configured
- [ ] Error monitoring setup
- [ ] Performance optimized
- [ ] Mobile testing completed

**Congratulations! Your Madrasa Management System is now live! 🎉**

### 📞 **Support & Updates**

For support:
1. Check documentation
2. Review error logs
3. Test step by step
4. Contact developer if needed

Regular updates:
- Security patches
- Feature enhancements
- Bug fixes
- Performance improvements

**Your complete madrasa management solution is ready! 🕌✨**