<?php
/**
 * Templates Page
 * Browse and use pre-built app templates
 */

include 'includes/header.php';

$templates = get_app_templates();
$categories = get_template_categories();
?>
<div class="templates">
    <div class="templates-header">
        <div class="container">
            <h1><i class="fas fa-template"></i> App Templates</h1>
            <p>Start quickly with professionally designed application templates</p>
        </div>
    </div>

    <div class="container">
        <!-- Category Filter -->
        <div class="category-filter">
            <button class="filter-btn active" data-category="all">All Templates</button>
            <?php foreach ($categories as $category): ?>
                <button class="filter-btn" data-category="<?php echo strtolower($category); ?>">
                    <?php echo ucfirst($category); ?>
                </button>
            <?php endforeach; ?>
        </div>

        <!-- Templates Grid -->
        <div class="templates-grid">
            <?php foreach ($templates as $template): ?>
                <div class="template-card" data-category="<?php echo strtolower($template['category']); ?>">
                    <div class="template-icon">
                        <i class="<?php echo htmlspecialchars($template['icon']); ?>"></i>
                    </div>
                    
                    <div class="template-content">
                        <h3><?php echo htmlspecialchars($template['name']); ?></h3>
                        <p><?php echo htmlspecialchars($template['description']); ?></p>
                        <div class="template-category"><?php echo htmlspecialchars($template['category']); ?></div>
                    </div>
                    
                    <div class="template-actions">
                        <button class="btn btn-primary" onclick="useTemplate('<?php echo htmlspecialchars($template['prompt_template']); ?>', '<?php echo htmlspecialchars($template['name']); ?>')">
                            <i class="fas fa-play"></i> Use Template
                        </button>
                    </div>
                </div>
            <?php endforeach; ?>
        </div>

        <!-- No Results Message -->
        <div class="no-results" style="display: none;">
            <i class="fas fa-search"></i>
            <h3>No templates found</h3>
            <p>Try selecting a different category or check back later for more templates.</p>
        </div>
    </div>
</div>

<style>
.templates {
    min-height: 100vh;
    background: #F8FAFC;
}

.templates-header {
    background: linear-gradient(135deg, #3B82F6 0%, #1D4ED8 100%);
    color: white;
    padding: 4rem 0 3rem;
    text-align: center;
}

.templates-header h1 {
    font-size: 3rem;
    margin-bottom: 1rem;
    font-weight: 700;
}

.templates-header h1 i {
    margin-right: 1rem;
}

.templates-header p {
    font-size: 1.25rem;
    opacity: 0.9;
}

.container {
    max-width: 1200px;
    margin: 0 auto;
    padding: 0 20px;
}

.category-filter {
    display: flex;
    gap: 1rem;
    margin: -3rem 0 3rem;
    justify-content: center;
    flex-wrap: wrap;
}

.filter-btn {
    padding: 0.75rem 1.5rem;
    border: 2px solid #E5E7EB;
    background: white;
    color: #6B7280;
    border-radius: 25px;
    cursor: pointer;
    transition: all 0.3s;
    font-weight: 500;
}

.filter-btn:hover,
.filter-btn.active {
    background: #3B82F6;
    color: white;
    border-color: #3B82F6;
}

.templates-grid {
    display: grid;
    grid-template-columns: repeat(auto-fill, minmax(350px, 1fr));
    gap: 2rem;
    margin: 2rem 0;
}

.template-card {
    background: white;
    border-radius: 16px;
    padding: 2rem;
    box-shadow: 0 4px 12px rgba(0,0,0,0.08);
    transition: all 0.3s;
    border: 2px solid transparent;
}

.template-card:hover {
    transform: translateY(-4px);
    box-shadow: 0 12px 24px rgba(0,0,0,0.15);
    border-color: #3B82F6;
}

.template-icon {
    width: 60px;
    height: 60px;
    background: linear-gradient(135deg, #3B82F6, #1D4ED8);
    border-radius: 12px;
    display: flex;
    align-items: center;
    justify-content: center;
    color: white;
    font-size: 1.5rem;
    margin-bottom: 1.5rem;
}

.template-content h3 {
    font-size: 1.5rem;
    color: #111827;
    margin-bottom: 0.75rem;
    font-weight: 600;
}

.template-content p {
    color: #6B7280;
    line-height: 1.6;
    margin-bottom: 1rem;
}

.template-category {
    display: inline-block;
    background: #EBF8FF;
    color: #1E40AF;
    padding: 0.25rem 0.75rem;
    border-radius: 20px;
    font-size: 0.75rem;
    font-weight: 500;
    margin-bottom: 1.5rem;
}

.template-actions {
    display: flex;
    gap: 1rem;
}

.btn {
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    padding: 0.75rem 1.5rem;
    border: none;
    border-radius: 8px;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.3s;
    text-decoration: none;
    font-size: 1rem;
    flex: 1;
    justify-content: center;
}

.btn-primary {
    background: linear-gradient(135deg, #3B82F6, #1D4ED8);
    color: white;
}

.btn-primary:hover {
    background: linear-gradient(135deg, #1D4ED8, #1E40AF);
    transform: translateY(-2px);
}

.no-results {
    text-align: center;
    padding: 4rem 2rem;
    background: white;
    border-radius: 16px;
    box-shadow: 0 4px 12px rgba(0,0,0,0.08);
    margin: 2rem 0;
}

.no-results i {
    font-size: 4rem;
    color: #D1D5DB;
    margin-bottom: 1.5rem;
}

.no-results h3 {
    font-size: 1.5rem;
    color: #6B7280;
    margin-bottom: 1rem;
}

.no-results p {
    color: #9CA3AF;
}

/* Template Card - Hidden by Default */
.template-card.hidden {
    display: none;
}

/* Responsive */
@media (max-width: 768px) {
    .templates-header h1 {
        font-size: 2rem;
    }
    
    .category-filter {
        margin: -2rem 0 2rem;
        justify-content: flex-start;
        overflow-x: auto;
        padding-bottom: 0.5rem;
    }
    
    .filter-btn {
        white-space: nowrap;
        flex-shrink: 0;
    }
    
    .templates-grid {
        grid-template-columns: 1fr;
    }
    
    .template-actions {
        flex-direction: column;
    }
}

/* Animation for filtered results */
.template-card {
    animation: fadeIn 0.3s ease-in-out;
}

@keyframes fadeIn {
    from {
        opacity: 0;
        transform: translateY(20px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}
</style>

<script>
function useTemplate(template, name) {
    // Redirect to generator page with template pre-filled
    const url = new URL('?page=generator', window.location.origin);
    url.searchParams.set('template', encodeURIComponent(template));
    url.searchParams.set('name', encodeURIComponent(name));
    window.location.href = url.toString();
}

// Category filter functionality
document.querySelectorAll('.filter-btn').forEach(btn => {
    btn.addEventListener('click', function() {
        // Update active button
        document.querySelectorAll('.filter-btn').forEach(b => b.classList.remove('active'));
        this.classList.add('active');
        
        const category = this.dataset.category;
        const templates = document.querySelectorAll('.template-card');
        
        // Filter templates
        let visibleCount = 0;
        templates.forEach(template => {
            if (category === 'all' || template.dataset.category === category) {
                template.classList.remove('hidden');
                visibleCount++;
            } else {
                template.classList.add('hidden');
            }
        });
        
        // Show/hide no results message
        const noResults = document.querySelector('.no-results');
        if (visibleCount === 0) {
            noResults.style.display = 'block';
        } else {
            noResults.style.display = 'none';
        }
    });
});
</script>

<?php
function get_app_templates() {
    global $db;
    
    try {
        $stmt = $db->query("SELECT * FROM templates WHERE is_active = 1 ORDER BY sort_order ASC");
        return $stmt->fetchAll();
    } catch (Exception $e) {
        log_error("Failed to get templates: " . $e->getMessage());
        return [];
    }
}

function get_template_categories() {
    global $db;
    
    try {
        $stmt = $db->query("SELECT DISTINCT category FROM templates WHERE is_active = 1 ORDER BY category");
        $categories = $stmt->fetchAll(PDO::FETCH_COLUMN);
        return $categories ?: ['Business', 'Creative', 'Personal', 'Content'];
    } catch (Exception $e) {
        log_error("Failed to get template categories: " . $e->getMessage());
        return ['Business', 'Creative', 'Personal', 'Content'];
    }
}
?>

<?php
include 'includes/footer.php';
?>