<?php
/**
 * Dashboard Page
 * Main landing page for logged-in users
 */

// Include header
include 'includes/header.php';

// Get user data
$user = get_user_by_id(get_current_user_id());

// Get recent projects
$recent_projects = get_recent_projects(get_current_user_id(), 6);

// Get project statistics
$stats = get_user_statistics(get_current_user_id());
?>
<div class="dashboard">
    <div class="dashboard-header">
        <div class="container">
            <h1>Welcome back, <?php echo htmlspecialchars($user['name'] ?? $user['email']); ?>!</h1>
            <p>Create amazing web applications with the power of AI</p>
        </div>
    </div>

    <div class="container">
        <!-- Statistics Cards -->
        <div class="stats-grid">
            <div class="stat-card">
                <div class="stat-icon">
                    <i class="fas fa-project-diagram"></i>
                </div>
                <div class="stat-content">
                    <h3><?php echo $stats['total_projects']; ?></h3>
                    <p>Total Projects</p>
                </div>
            </div>
            
            <div class="stat-card">
                <div class="stat-icon">
                    <i class="fas fa-check-circle"></i>
                </div>
                <div class="stat-content">
                    <h3><?php echo $stats['completed_projects']; ?></h3>
                    <p>Completed</p>
                </div>
            </div>
            
            <div class="stat-card">
                <div class="stat-icon">
                    <i class="fas fa-clock"></i>
                </div>
                <div class="stat-content">
                    <h3><?php echo $stats['draft_projects']; ?></h3>
                    <p>In Progress</p>
                </div>
            </div>
            
            <div class="stat-card">
                <div class="stat-icon">
                    <i class="fas fa-download"></i>
                </div>
                <div class="stat-content">
                    <h3><?php echo $stats['total_downloads']; ?></h3>
                    <p>Downloads</p>
                </div>
            </div>
        </div>

        <!-- Quick Actions -->
        <div class="quick-actions">
            <h2>Quick Actions</h2>
            <div class="actions-grid">
                <a href="?page=generator" class="action-card primary">
                    <div class="action-icon">
                        <i class="fas fa-plus-circle"></i>
                    </div>
                    <h3>Create New App</h3>
                    <p>Generate a new web application using AI</p>
                </a>
                
                <a href="?page=templates" class="action-card">
                    <div class="action-icon">
                        <i class="fas fa-template"></i>
                    </div>
                    <h3>Browse Templates</h3>
                    <p>Start with pre-built app templates</p>
                </a>
                
                <a href="?page=projects" class="action-card">
                    <div class="action-icon">
                        <i class="fas fa-folder"></i>
                    </div>
                    <h3>My Projects</h3>
                    <p>View and manage your applications</p>
                </a>
                
                <a href="?page=settings" class="action-card">
                    <div class="action-icon">
                        <i class="fas fa-cog"></i>
                    </div>
                    <h3>Settings</h3>
                    <p>Configure your account and preferences</p>
                </a>
            </div>
        </div>

        <!-- Recent Projects -->
        <div class="recent-projects">
            <div class="section-header">
                <h2>Recent Projects</h2>
                <a href="?page=projects" class="btn btn-secondary">View All</a>
            </div>
            
            <?php if (empty($recent_projects)): ?>
                <div class="empty-state">
                    <i class="fas fa-project-diagram"></i>
                    <h3>No projects yet</h3>
                    <p>Create your first AI-powered application to get started</p>
                    <a href="?page=generator" class="btn btn-primary">Create Your First App</a>
                </div>
            <?php else: ?>
                <div class="projects-grid">
                    <?php foreach ($recent_projects as $project): ?>
                        <div class="project-card">
                            <div class="project-header">
                                <h4><?php echo htmlspecialchars($project['title']); ?></h4>
                                <span class="status-badge status-<?php echo $project['status']; ?>">
                                    <?php echo ucfirst($project['status']); ?>
                                </span>
                            </div>
                            
                            <p class="project-description">
                                <?php echo htmlspecialchars(substr($project['description'] ?? '', 0, 100)); ?>
                                <?php echo strlen($project['description'] ?? '') > 100 ? '...' : ''; ?>
                            </p>
                            
                            <div class="project-meta">
                                <span class="project-type"><?php echo ucfirst($project['app_type']); ?></span>
                                <span class="project-date"><?php echo time_ago($project['created_at']); ?></span>
                            </div>
                            
                            <div class="project-actions">
                                <?php if ($project['status'] === 'complete'): ?>
                                    <a href="?page=export&project=<?php echo $project['id']; ?>" class="btn btn-sm btn-primary">
                                        <i class="fas fa-download"></i> Export
                                    </a>
                                    <a href="?page=projects&action=preview&project=<?php echo $project['id']; ?>" class="btn btn-sm btn-secondary">
                                        <i class="fas fa-eye"></i> Preview
                                    </a>
                                <?php elseif ($project['status'] === 'generating'): ?>
                                    <button class="btn btn-sm btn-secondary" disabled>
                                        <i class="fas fa-spinner fa-spin"></i> Generating...
                                    </button>
                                <?php else: ?>
                                    <a href="?page=projects&action=edit&project=<?php echo $project['id']; ?>" class="btn btn-sm btn-primary">
                                        <i class="fas fa-edit"></i> Edit
                                    </a>
                                <?php endif; ?>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            <?php endif; ?>
        </div>

        <!-- Getting Started Guide -->
        <div class="getting-started">
            <h2>Getting Started</h2>
            <div class="guide-steps">
                <div class="step">
                    <div class="step-number">1</div>
                    <div class="step-content">
                        <h4>Describe Your App</h4>
                        <p>Tell the AI what kind of application you want to create. Be specific about features and design.</p>
                    </div>
                </div>
                
                <div class="step">
                    <div class="step-number">2</div>
                    <div class="step-content">
                        <h4>AI Generation</h4>
                        <p>Our AI will analyze your request and generate a complete web application with HTML, CSS, and JavaScript.</p>
                    </div>
                </div>
                
                <div class="step">
                    <div class="step-number">3</div>
                    <div class="step-content">
                        <h4>Customize & Export</h4>
                        <p>Review the generated code, make any customizations, and export it for deployment to any hosting service.</p>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<style>
.dashboard {
    min-height: 100vh;
    background: #F8FAFC;
}

.dashboard-header {
    background: linear-gradient(135deg, #3B82F6 0%, #1D4ED8 100%);
    color: white;
    padding: 4rem 0 3rem;
    text-align: center;
}

.dashboard-header h1 {
    font-size: 3rem;
    margin-bottom: 1rem;
    font-weight: 700;
}

.dashboard-header p {
    font-size: 1.25rem;
    opacity: 0.9;
}

.container {
    max-width: 1200px;
    margin: 0 auto;
    padding: 0 20px;
}

/* Statistics Grid */
.stats-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
    gap: 2rem;
    margin: -3rem 0 3rem;
}

.stat-card {
    background: white;
    border-radius: 16px;
    padding: 2rem;
    box-shadow: 0 10px 25px rgba(0,0,0,0.1);
    display: flex;
    align-items: center;
    gap: 1.5rem;
    transition: transform 0.3s;
}

.stat-card:hover {
    transform: translateY(-4px);
}

.stat-icon {
    width: 60px;
    height: 60px;
    background: linear-gradient(135deg, #3B82F6, #1D4ED8);
    border-radius: 12px;
    display: flex;
    align-items: center;
    justify-content: center;
    color: white;
    font-size: 1.5rem;
}

.stat-content h3 {
    font-size: 2.5rem;
    font-weight: 700;
    color: #111827;
    margin-bottom: 0.5rem;
}

.stat-content p {
    color: #6B7280;
    font-weight: 500;
}

/* Quick Actions */
.quick-actions {
    margin: 3rem 0;
}

.quick-actions h2 {
    font-size: 2rem;
    margin-bottom: 2rem;
    color: #111827;
}

.actions-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
    gap: 2rem;
}

.action-card {
    background: white;
    border-radius: 16px;
    padding: 2rem;
    box-shadow: 0 4px 12px rgba(0,0,0,0.08);
    text-decoration: none;
    color: inherit;
    transition: all 0.3s;
    border: 2px solid transparent;
}

.action-card:hover {
    transform: translateY(-4px);
    box-shadow: 0 12px 24px rgba(0,0,0,0.15);
    text-decoration: none;
    color: inherit;
}

.action-card.primary {
    background: linear-gradient(135deg, #3B82F6, #1D4ED8);
    color: white;
    border-color: #3B82F6;
}

.action-icon {
    width: 50px;
    height: 50px;
    background: #F3F4F6;
    border-radius: 10px;
    display: flex;
    align-items: center;
    justify-content: center;
    margin-bottom: 1rem;
    color: #3B82F6;
    font-size: 1.25rem;
}

.action-card.primary .action-icon {
    background: rgba(255,255,255,0.2);
    color: white;
}

.action-card h3 {
    font-size: 1.25rem;
    margin-bottom: 0.5rem;
    font-weight: 600;
}

.action-card p {
    color: #6B7280;
    line-height: 1.6;
}

.action-card.primary p {
    color: rgba(255,255,255,0.9);
}

/* Recent Projects */
.recent-projects {
    margin: 4rem 0;
}

.section-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 2rem;
}

.section-header h2 {
    font-size: 2rem;
    color: #111827;
}

.empty-state {
    text-align: center;
    padding: 4rem 2rem;
    background: white;
    border-radius: 16px;
    box-shadow: 0 4px 12px rgba(0,0,0,0.08);
}

.empty-state i {
    font-size: 4rem;
    color: #D1D5DB;
    margin-bottom: 1.5rem;
}

.empty-state h3 {
    font-size: 1.5rem;
    color: #6B7280;
    margin-bottom: 1rem;
}

.empty-state p {
    color: #9CA3AF;
    margin-bottom: 2rem;
}

.projects-grid {
    display: grid;
    grid-template-columns: repeat(auto-fill, minmax(350px, 1fr));
    gap: 2rem;
}

.project-card {
    background: white;
    border-radius: 12px;
    padding: 1.5rem;
    box-shadow: 0 4px 12px rgba(0,0,0,0.08);
    transition: transform 0.3s;
}

.project-card:hover {
    transform: translateY(-2px);
}

.project-header {
    display: flex;
    justify-content: space-between;
    align-items: start;
    margin-bottom: 1rem;
}

.project-header h4 {
    font-size: 1.25rem;
    color: #111827;
    margin: 0;
}

.status-badge {
    padding: 0.25rem 0.75rem;
    border-radius: 20px;
    font-size: 0.75rem;
    font-weight: 600;
    text-transform: uppercase;
}

.status-draft { background: #FEF3C7; color: #92400E; }
.status-generating { background: #DBEAFE; color: #1E40AF; }
.status-complete { background: #D1FAE5; color: #065F46; }
.status-error { background: #FEE2E2; color: #991B1B; }

.project-description {
    color: #6B7280;
    margin-bottom: 1rem;
    line-height: 1.6;
}

.project-meta {
    display: flex;
    justify-content: space-between;
    margin-bottom: 1rem;
    font-size: 0.875rem;
    color: #9CA3AF;
}

.project-actions {
    display: flex;
    gap: 0.5rem;
}

/* Getting Started */
.getting-started {
    margin: 4rem 0;
    background: white;
    border-radius: 16px;
    padding: 3rem;
    box-shadow: 0 4px 12px rgba(0,0,0,0.08);
}

.getting-started h2 {
    font-size: 2rem;
    margin-bottom: 2rem;
    color: #111827;
    text-align: center;
}

.guide-steps {
    display: flex;
    flex-direction: column;
    gap: 2rem;
}

.step {
    display: flex;
    gap: 2rem;
    align-items: start;
}

.step-number {
    width: 40px;
    height: 40px;
    background: linear-gradient(135deg, #3B82F6, #1D4ED8);
    color: white;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    font-weight: 600;
    flex-shrink: 0;
}

.step-content h4 {
    font-size: 1.25rem;
    color: #111827;
    margin-bottom: 0.5rem;
}

.step-content p {
    color: #6B7280;
    line-height: 1.6;
}

/* Responsive */
@media (max-width: 768px) {
    .dashboard-header h1 {
        font-size: 2rem;
    }
    
    .dashboard-header p {
        font-size: 1rem;
    }
    
    .stats-grid {
        grid-template-columns: 1fr;
        margin: -2rem 0 2rem;
    }
    
    .actions-grid {
        grid-template-columns: 1fr;
    }
    
    .projects-grid {
        grid-template-columns: 1fr;
    }
    
    .section-header {
        flex-direction: column;
        gap: 1rem;
        text-align: center;
    }
    
    .step {
        flex-direction: column;
        text-align: center;
    }
    
    .project-actions {
        flex-direction: column;
    }
}

/* Button Styles */
.btn {
    display: inline-block;
    padding: 0.75rem 1.5rem;
    border: none;
    border-radius: 8px;
    text-decoration: none;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.3s;
    font-size: 0.875rem;
    text-align: center;
}

.btn-sm {
    padding: 0.5rem 1rem;
    font-size: 0.75rem;
}

.btn-primary {
    background: #3B82F6;
    color: white;
}

.btn-primary:hover {
    background: #1D4ED8;
    text-decoration: none;
    color: white;
}

.btn-secondary {
    background: #6B7280;
    color: white;
}

.btn-secondary:hover {
    background: #4B5563;
    text-decoration: none;
    color: white;
}
</style>

<?php
// Include footer
include 'includes/footer.php';
?>